/***************************************************************************
*
* Copyright (C) 2013 DENSO CORPORATION and Robert Bosch Car Multimedia Gmbh
*
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*        http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
****************************************************************************/
#include "WindowSystems/wayland-1.0-server-imports.h"
#include "TextureBinders/WaylandGcoSurf.h"
#include "PlatformSurfaces/GcoSurfWaylandPlatformSurface.h"
#include "Log.h"
#include "wayland-server.h"
#include "config.h"
#include <string.h>

WaylandGcoSurf::WaylandGcoSurf(gcoHAL gcoHal, gco2D gco2d, struct wl_display* wlDisplay,Scene* pScene)
: m_gcoHal(gcoHal)
, m_gco2dEngine(gco2d)
, m_wlDisplay(wlDisplay)
, m_pScene(pScene)
{
}

WaylandGcoSurf::~WaylandGcoSurf()
{
}

void
WaylandGcoSurf::correctSourceRectangle(Surface* surface, int width, int height)
{
    bool rectCorrected = false;
    FloatRectangle newregion;
    FloatRectangle region = surface->getTargetSourceRegion();

    newregion = region;

    if(region.x < 0)
    {
        newregion.x = 0;
        rectCorrected = true;
    }

    if(region.x > (width-1))
    {
        rectCorrected = true;
        newregion.x = width-1;
    }

    if((region.x + region.width) > width)
    {
        newregion.width = width - region.x;
        rectCorrected = true;
    }

    if(region.y < 0)
    {
        newregion.y = 0;
        rectCorrected = true;
    }

    if(region.y > (height-1))
    {
        newregion.height = height - 1;
        rectCorrected = true;
    }

    if((region.y + region.height) > height)
    {
        newregion.height = height - region.y;
        rectCorrected = true;
    }

    if(rectCorrected)
    {
    	FloatRectangle currCorrectedRegion = surface->getCorrectedSrcRegion();
        if (   (currCorrectedRegion.width != newregion.width  )
            || (currCorrectedRegion.height != newregion.height)
            || (currCorrectedRegion.x != newregion.x          )
            || (currCorrectedRegion.y != newregion.y          ) )
        {
        	LOG_WARNING("WaylandGcoSurf",
        	            "Source Rectangle corrected to X:" << newregion.x <<
        	            "Y:" << newregion.y <<
        	            "width:" << newregion.width <<
        	            "height:" << newregion.height);
        }
    }
	surface->setCorrectedSrcRegion(newregion);
}

bool WaylandGcoSurf::bindSurfaceTexture(Surface* surface)
{
    GcoSurfWaylandPlatformSurface* nativeSurface = (GcoSurfWaylandPlatformSurface*)surface->platform;
    if (!(nativeSurface && nativeSurface->isReadyForRendering()))
    {
        LOG_WARNING("WaylandGcoSurf",
                    "nativeSurface isn't ready, "
                    "surface=" << surface << ", "
                    "surface ID=" << surface->getID() << ", "
                    "native display=" << m_wlDisplay << ", "
                    "native handle=" << surface->getNativeContent() << ", "
                    "native buffer=" << surface->getNativeBuffer());
        return false;
    }

    gctUINT width = 0;
    gctUINT height = 0;
    gctINT stride = 0;
    gceSURF_FORMAT format;
    gctUINT32 phyAddr = 0;
    gctPOINTER logAddr = NULL;
    gceSTATUS status = gcvSTATUS_OK;
    FloatRectangle region;
    gceTILING tiling;

    status = gcoSURF_GetAlignedSize(nativeSurface->m_surface,
                                    &width, &height, &stride);
    if (gcvSTATUS_OK != status)
    {
        LOG_WARNING("WaylandGcoSurf",
                    "Failed to get aligned size, "
                    "surface=" << surface << ", "
                    "surface ID=" << surface->getID() << ", "
                    "native display=" << m_wlDisplay << ", "
                    "native handle=" << surface->getNativeContent() << ", "
                    "native buffer=" << surface->getNativeBuffer());
        return false;
    }
    status = gcoSURF_GetFormat(nativeSurface->m_surface, gcvNULL, &format);
    if (gcvSTATUS_OK != status)
    {
        LOG_WARNING("WaylandGcoSurf",
                    "Failed to get format, "
                    "surface=" << surface << ", "
                    "surface ID=" << surface->getID() << ", "
                    "native display=" << m_wlDisplay << ", "
                    "native handle=" << surface->getNativeContent() << ", "
                    "native buffer=" << surface->getNativeBuffer());
        return false;
    }
	//Obtain a source rectangle limited by actual buffer width and height.
    correctSourceRectangle(surface, width, height);
    region = surface->getCorrectedSrcRegion();

    /* If this is a YUV surface, we will not go through the usual gco2D path
     * for blitting. No point in setting the sources here so bail out now. */
    switch (format) {
    case gcvSURF_YUY2:
    case gcvSURF_UYVY:
    case gcvSURF_YV12:
    case gcvSURF_I420:
    case gcvSURF_NV12:
    case gcvSURF_NV21:
    case gcvSURF_NV16:
    case gcvSURF_NV61:
    case gcvSURF_YVYU:
    case gcvSURF_VYUY:
        LOG_DEBUG("WaylandGcoSurf",
                  "YUV surface, sources not set here so returning");
        return true;

    default:    // Prevent compiler warning.
        break;
    }

    status = gcoSURF_Lock(nativeSurface->m_surface, &phyAddr, (gctPOINTER*)&logAddr);
    if (gcvSTATUS_OK != status)
    {
        LOG_WARNING("WaylandGcoSurf",
                    "Failed to lock surface, "
                    "surface=" << surface << ", "
                    "surface ID=" << surface->getID() << ", "
                    "native display=" << m_wlDisplay << ", "
                    "native handle=" << surface->getNativeContent() << ", "
                    "native buffer=" << surface->getNativeBuffer());
        return false;
    }
    else
    {
        LOG_DEBUG("WaylandGcoSurf",
                  "Surface locked, "
                  "surface=" << surface << ", "
                  "surface ID=" << surface->getID() << ", "
                  "containing layer ID="
                  << surface->getContainingLayerId() << ", "
                  "width=" << width << ", "
                  "height=" << height << ", "
                  "stride=" << stride << ", "
                  "native display=" << m_wlDisplay << ", "
                  "native surface=" << nativeSurface->m_surface << ", "
                  "native handle=" << surface->getNativeContent() << ", "
                  "native buffer=" << surface->getNativeBuffer() << ", "
                  "wayland platform surface=" << nativeSurface << ", "
                  "format=" << format << ", "
                  "physical address=" << &phyAddr << ", "
                  "logical address=" << &logAddr);
    }

    gcoSURF_GetTiling(nativeSurface->m_surface, &tiling);

    LOG_INFO("WaylandGcoSurf",
              "Tiling Format, "
              "surface=" << surface << ", "
              "surface ID=" << surface->getID() << ", "
              "Tiling=" << tiling);

    if(gcoHAL_IsFeatureAvailable(m_gcoHal, gcvFEATURE_2D_TILING) != gcvTRUE &&
		    (tiling > gcvLINEAR))
    {
        LOG_WARNING("WaylandGcoSurf",
                        "Tiling not supported, but"
                        "surface ID=" << surface->getID() << " is in tiled format");
        gcoSURF_Unlock(nativeSurface->m_surface, (gctPOINTER)logAddr);
        return false;
    }

    status = gco2D_SetGenericSource(m_gco2dEngine, &phyAddr, 1,
                    (gctUINT32_PTR)&stride, 1, tiling, format, gcvSURF_0_DEGREE,
                    width, height);

    if (gcvSTATUS_OK != status)
    {
            LOG_WARNING("WaylandGcoSurf", "Failed to set color surface,"
                            "surface=" << surface << ", "
                            "surface ID=" << surface->getID() << ", "
                            "containing layer ID="
                            << surface->getContainingLayerId() << ", "
                            "width=" << width << ", "
                            "height=" << height << ", "
                            "stride=" << stride << ", "
                            "native display=" << m_wlDisplay << ", "
                            "native surface=" << nativeSurface->m_surface << ", "
                            "native handle=" << surface->getNativeContent() << ", "
                            "native buffer=" << surface->getNativeBuffer() << ", "
                            "wayland platform surface=" << nativeSurface << ", "
                            "format=" << format << ", "
                            "physical address=" << &phyAddr << ", "
                            "logical address=" << &logAddr);
            gcoSURF_Unlock(nativeSurface->m_surface, (gctPOINTER)logAddr);
            return false;
    }

    gcsRECT rect = {0, 0, 0, 0};

    rect.left   = region.x;
    rect.top    = region.y;
    rect.right  = region.width + region.x;
    rect.bottom = region.height + region.y;

    status = gco2D_SetSource(m_gco2dEngine, &rect);
    if (gcvSTATUS_OK != status)
    {
        LOG_WARNING("WaylandGcoSurf",
                    "Failed to set source, "
                    "surface=" << surface << ", "
                    "surface ID=" << surface->getID() << ", "
                    "containing layer ID="
                    << surface->getContainingLayerId() << ", "
                    "width=" << width << ", "
                    "height=" << height << ", "
                    "stride=" << stride << ", "
                    "native display=" << m_wlDisplay << ", "
                    "native surface=" << nativeSurface->m_surface << ", "
                    "native handle=" << surface->getNativeContent() << ", "
                    "native buffer=" << surface->getNativeBuffer() << ", "
                    "wayland platform surface=" << nativeSurface << ", "
                    "format=" << format << ", "
                    "physical address=" << &phyAddr << ", "
                    "logical address=" << &logAddr << ", "
                    "srcRect, left=" << rect.left << ", "
                    "top=" << rect.top << ", "
                    "right=" << rect.right << ", "
                    "bottom=" << rect.bottom);
        gcoSURF_Unlock(nativeSurface->m_surface, (gctPOINTER)logAddr);
        return false;
    }

    status = gcoSURF_Unlock(nativeSurface->m_surface, (gctPOINTER)logAddr);
    if (gcvSTATUS_OK != status)
    {
        LOG_WARNING("WaylandGcoSurf",
                    "Failed to unlock surface, "
                    "wayland platform surface=" << nativeSurface << ", "
                    "native surface=" << nativeSurface->m_surface << ", "
                    "physical address=" << &phyAddr << ", "
                    "logical address=" << &logAddr);
    }
    else
    {
        LOG_DEBUG("WaylandGcoSurf",
                  "Surface unlocked, "
                  "surface=" << surface << ", "
                  "surface ID=" << surface->getID() << ", "
                  "srcRect, left=" << rect.left << ", "
                  "top=" << rect.top << ", "
                  "right=" << rect.right << ", "
                  "bottom=" << rect.bottom);
    }

    return true;
}

bool WaylandGcoSurf::unbindSurfaceTexture(Surface* surface)
{
    LOG_DEBUG("WaylandGcoSurf",
              "Called, surface=" << surface
              << ", surface ID=" << surface->getID()
              << ", containing layer ID="
              << surface->getContainingLayerId());
    surface = surface;
    return true;
}

void WaylandGcoSurf::createClientBuffer(Surface* surface)
{
    gceSTATUS status = gcvSTATUS_OK;

    LOG_DEBUG("WaylandGcoSurf",
              "Called, creating client buffer, "
              "surface=" << surface << ", "
              "surface ID=" << surface->getID() << ", "
              "native display=" << m_wlDisplay << ", "
              "native handle=" << surface->getNativeContent() << ", "
              "native buffer=" << surface->getNativeBuffer());

    GcoSurfWaylandPlatformSurface* nativeSurface = (GcoSurfWaylandPlatformSurface*)surface->platform;

    if (NULL == nativeSurface)
    {
        LOG_WARNING("WaylandGcoSurf",
                    "nativeSurface is NULL in createClientBuffer, "
                    "surface=" << surface << ", "
                    "surface ID=" << surface->getID());
        return;
    }

    struct lm_wl_buffer* lm_buffer = (struct lm_wl_buffer*)surface->getNativeBuffer();

    if (NULL == lm_buffer || NULL == lm_buffer->resource)
    {
        LOG_WARNING("WaylandGcoSurf",
                    "surface native content is NULL, "
                    "surface=" << surface << ", "
                    "surface ID=" << surface->getID());
        return;
    }
    struct wl_shm_buffer *buffer = wl_shm_buffer_get(lm_buffer->resource);
    if (!buffer)
    {

        LOG_DEBUG("WaylandGcoSurf",
                  "No buffer, creating and setting native surface, "
                  "surface ID=" << surface->getID() << ", "
                  "wayland platform surface=" << nativeSurface);
        gcsWL_VIV_BUFFER* wlVivBuffer = (gcsWL_VIV_BUFFER*)wl_resource_get_user_data(lm_buffer->resource);
        nativeSurface->m_surface = wlVivBuffer->surface;
    }
    else
    {
        gceSURF_FORMAT format;
        gctUINT alignedWidth = 0;
        gctUINT alignedHeight = 0;
        gctINT alignedStride = 0;
        gctUINT32 phyAddr = 0;
        gctPOINTER logAddr = NULL;

        gctPOINTER srcLogAddr = (gctPOINTER)wl_shm_buffer_get_data((struct wl_shm_buffer *)buffer);
        if (wl_shm_buffer_get_format(buffer) == WL_SHM_FORMAT_XRGB8888)
        {
            format = gcvSURF_X8R8G8B8;
        }
        else
        {
            format = gcvSURF_A8R8G8B8;
        }

        gctUINT shm_buffer_width = wl_shm_buffer_get_width(buffer);
        gctUINT shm_buffer_height = wl_shm_buffer_get_height(buffer);

        if (NULL != nativeSurface->m_surface)
        {
            gctUINT alignmentW = 0;
            gceSURF_FORMAT gco_format;
            gctUINT shm_buffer_width_aligned;

            LOG_DEBUG("WaylandGcoSurf",
                      "Buffer exists, native surface already exists, "
                      "check dimensions before destroying of native surface, "
                      "wayland platform surface=" << nativeSurface << ", "
                      "surface ID=" << surface->getID());
            /*check alignment and format of the surface before destroying*/

            status = gcoSURF_GetAlignedSize(nativeSurface->m_surface, &alignedWidth, &alignedHeight, &alignedStride);
            if (gcvSTATUS_OK != status)
            {
                LOG_WARNING("WaylandGcoSurf",
                            "Failed to get aligned size for SHM"
                            " in createClientBuffer, "
                            "wayland platform surface=" << nativeSurface << ", "
                            "native surface=" << nativeSurface->m_surface << ", "
                            "surface ID=" << surface->getID() << ", "
                            "alignedWidth=" << alignedWidth << ", "
                            "alignedHeight=" << alignedHeight << ", "
                            "alignedStride=" << alignedStride);
                return;
            }

            status = gcoSURF_GetFormat(nativeSurface->m_surface, gcvNULL, &gco_format);
            if (gcvSTATUS_OK != status)
            {
                LOG_WARNING("WaylandGcoSurf",
                            "Failed to get format, "
                            "surface=" << surface << ", "
                            "surface ID=" << surface->getID() << ", "
                            "native display=" << m_wlDisplay << ", "
                            "native handle=" << surface->getNativeContent() << ", "
                            "native buffer=" << surface->getNativeBuffer());
                return;
            }

            /*GPU driver doesn't report a proper Height alignment, therefore ignoring*/
            status =  gcoSURF_GetAlignment(gcvSURF_BITMAP, gco_format, NULL, &alignmentW, NULL);
            if (gcvSTATUS_OK != status)
            {
                LOG_ERROR("WaylandGcoSurf","error by gcoSURF_GetAlignment"
                        "required to align the drm buffer size, status: "
                        << status);
                return;
            }

            shm_buffer_width_aligned = gcmALIGN_NP2(shm_buffer_width,alignmentW);

            if ((gco_format == format) &&
                    (alignedWidth == shm_buffer_width_aligned) &&
                    (alignedHeight >= shm_buffer_height))
            {
                LOG_DEBUG("WaylandGcoSurf",
                        "reuse already allocated gcoSURF");
            }
            else
            {
                gcoSURF_Destroy(nativeSurface->m_surface);
                nativeSurface->m_surface = NULL;
            }
        }

        if (NULL == nativeSurface->m_surface)
        {
            status = gcoSURF_Construct(m_gcoHal,
                                    shm_buffer_width,
                                    shm_buffer_height,
                                    1, gcvSURF_BITMAP, format,
                                    gcvPOOL_DEFAULT, &nativeSurface->m_surface);
            if (gcvSTATUS_OK != status)
            {
                LOG_WARNING("WaylandGcoSurf",
                          "Failed to create gcoSURF "
                          "for SHM in createClientBuffer"
                          "wayland platform surface=" << nativeSurface << ", "
                          "surface ID=" << surface->getID() << ", "
                          "shm_buffer_width=" << shm_buffer_width << ", "
                          "shm_buffer_height=" << shm_buffer_height << ", "
                          "format code=" << format);
                return;
            }
            else
            {
                LOG_DEBUG("WaylandGcoSurf",
                          "Created gcoSURF for SHM in createClientBuffer, "
                          "wayland platform surface=" << nativeSurface << ", "
                          "native surface=" << nativeSurface->m_surface << ", "
                          "surface ID=" << surface->getID() << ", "
                          "shm_buffer_width=" << shm_buffer_width << ", "
                          "shm_buffer_height=" << shm_buffer_height << ", "
                          "format code=" << format);
            }

            status = gcoSURF_GetAlignedSize(nativeSurface->m_surface, &alignedWidth, &alignedHeight, &alignedStride);
            if (gcvSTATUS_OK != status)
            {
                LOG_WARNING("WaylandGcoSurf",
                            "Failed to get aligned size for SHM"
                            " in createClientBuffer, "
                            "wayland platform surface=" << nativeSurface << ", "
                            "native surface=" << nativeSurface->m_surface << ", "
                            "surface ID=" << surface->getID() << ", "
                            "alignedWidth=" << alignedWidth << ", "
                            "alignedHeight=" << alignedHeight << ", "
                            "alignedStride=" << alignedStride);
                return;
            }
        }

        status = gcoSURF_Lock(nativeSurface->m_surface, &phyAddr, &logAddr);
        if (gcvSTATUS_OK != status)
        {
            LOG_WARNING("WaylandGcoSurf",
                        "Failed to lock surface for SHM"
                        " in createClientBuffer, "
                        "wayland platform surface=" << nativeSurface << ", "
                        "surface ID=" << surface->getID());
            return;
        }
        else
        {
            LOG_DEBUG("WaylandGcoSurf",
                      "Locked surface for SHM in createClientBuffer, "
                      "wayland platform surface=" << nativeSurface << ", "
                      "native surface=" << nativeSurface->m_surface << ", "
                      "surface ID=" << surface->getID() << ", "
                      "physical address=" << &phyAddr << ", "
                      "logical address=" << &logAddr << ", "
                      "srcLogAddr=" << srcLogAddr);
        }

        wl_shm_buffer_begin_access((struct wl_shm_buffer *)buffer);
        if ((alignedWidth == shm_buffer_width) && (alignedHeight == shm_buffer_height))
        {
            int srcStride = wl_shm_buffer_get_stride((struct wl_shm_buffer *)buffer);
            LOG_DEBUG("WaylandGcoSurf",
                      "Aligned width/height match shm_buffer "
                      "worked out stride and copy, "
                      "srcStride=" << srcStride);
            memcpy(logAddr, srcLogAddr, srcStride * shm_buffer_height);
        }
        else
        {
            LOG_DEBUG("WaylandGcoSurf",
                      "Aligned width/height don't match shm_buffer "
                      "scale and copy, "
                      "alignedWidth=" << alignedWidth);

            unsigned char* srcAddr = 0;
            unsigned char* dstAddr = 0;
            unsigned int cnt = 0;
            for (; cnt < shm_buffer_height; cnt++)
            {
                srcAddr = (unsigned char*)srcLogAddr + ((cnt * shm_buffer_width)<<2);
                dstAddr = (unsigned char*)logAddr + ((cnt * alignedWidth)<<2);
                memcpy(dstAddr, srcAddr, shm_buffer_width<<2);
            }
        }
        wl_shm_buffer_end_access((struct wl_shm_buffer *)buffer);

        status = gcoSURF_Unlock(nativeSurface->m_surface, (gctPOINTER)logAddr);
        if (gcvSTATUS_OK != status)
        {
            LOG_WARNING("WaylandGcoSurf",
                        "Failed to unlock surface, "
                        "wayland platform surface=" << nativeSurface << ", "
                        "surface ID=" << surface->getID() << ", "
                        "physical address=" << &phyAddr << ", "
                        "logical address=" << &logAddr << ", "
                        "srcLogAddr=" << srcLogAddr);
        }
    }
}

PlatformSurface* WaylandGcoSurf::createPlatformSurface(Surface* surface)
{
    LOG_DEBUG("WaylandGcoSurf",
              "Called, surface=" << surface
              << ", surface ID=" << surface->getID()
              << ", containing layer ID="
              << surface->getContainingLayerId());

    return new GcoSurfWaylandPlatformSurface(surface);
}

void WaylandGcoSurf::destroyClientBuffer(Surface* surface)
{
    LOG_DEBUG("WaylandGcoSurf",
              "Called, surface=" << surface
              << ", surface ID=" << surface->getID()
              << ", containing layer ID="
              << surface->getContainingLayerId());

    GcoSurfWaylandPlatformSurface* nativeSurface = (GcoSurfWaylandPlatformSurface*)surface->platform;

    if (NULL == nativeSurface)
    {
        LOG_WARNING("WaylandGcoSurf",
                    "nativeSurface is NULL in destroyClientBuffer, "
                    "surface ID=" << surface->getID() << ", "
                    "containing layer ID="
                    << surface->getContainingLayerId());
        return;
    }
    struct lm_wl_buffer* buffer = (struct lm_wl_buffer*)surface->getNativeBuffer();
    if ((NULL != buffer) && (NULL != buffer->resource) && wl_shm_buffer_get(buffer->resource))
    {
        LOG_DEBUG("WaylandGLESTexture",
                  "wayland platform surface buffering present, "
                  "buffer=" << buffer);

        if (nativeSurface && nativeSurface->m_surface)
        {
            LOG_DEBUG("WaylandGcoSurf",
                      "Destroying native surface="
                      << nativeSurface->m_surface);
            gcoSURF_Destroy(nativeSurface->m_surface);
            nativeSurface->m_surface = NULL;
        }
    }
    else
    {
        if (nativeSurface && nativeSurface->m_surface)
        {
            LOG_DEBUG("WaylandGcoSurf",
                      "wayland and native platform surfaces set, "
                      "resetting to NULL");
            nativeSurface->m_surface = NULL;
        }
    }
}
